#!/bin/bash

# set -e - 遇到错误立即退出
# set -u - 使用未定义的变量时报错
# set -x - 显示执行的命令（调试用）
# set -o pipefail - 管道中任何命令失败都返回失败
set -e

# 检查容器状态
# Args:
#   $1: 容器名称（支持部分匹配，如 "python" 会匹配包含 python 的容器名）
#   $2: 等待时间（可选，默认 5 秒）
# Returns:
#   0: 容器运行正常
#   1: 容器状态异常或未运行
checkContainerStatus() {
    local container_name="$1"
    local wait_seconds="${2:-5}"  # 默认等待 5 秒，可通过第二个参数自定义

    if [ -z "$container_name" ]; then
        echo "ERROR: 容器名称不能为空"
        return 1
    fi

    echo "等待 ${wait_seconds} 秒，检查容器状态: ${container_name}"
    sleep "${wait_seconds}"

    # 获取容器状态（支持部分匹配）
    status=$(docker ps --filter "name=${container_name}" --format "{{.Status}}" 2>/dev/null)

    # 检查容器是否存在
    if [ -z "$status" ]; then
        echo "❌ 容器 ${container_name} 未运行或不存在"
        echo "   请检查容器是否已启动: docker ps -a | grep ${container_name}"
        return 1
    fi

    echo "容器状态: ${status}"

    # 检查状态是否以 "Up Less than" 开头（可能启动失败）
    if echo "$status" | grep -qE "^Up Less than [0-9]+ second"; then
        echo "!!! 容器 ${container_name} 启动可能失败, 请检查 !!!"
        echo "查看容器 ${container_name} 最后100行日志:"
        sh docker-compose-logs.sh --tail=100 ${container_name}
        return 1
    else
        echo "容器 ${container_name} 状态正常"
        return 0
    fi
}

echo "远端服务器执行发布脚本 ${REMOTE_APP_PATH}"

cd ${REMOTE_APP_PATH}/

# 检查目录是否存在
if [ -d "${REMOTE_PROJECT_NAME}" ]; then
    echo "目录 ${REMOTE_PROJECT_NAME} 已存在，删除文件（保留 *.ipdb, *.awdb, *.dat）"
    # 删除目录下所有文件，但排除 *.ipdb, *.awdb, *.dat 文件
    find ${REMOTE_PROJECT_NAME} -type f ! -name "*.ipdb" ! -name "*.awdb" ! -name "*.dat" -delete
    # 删除空目录（保留包含保留文件的目录）
    find ${REMOTE_PROJECT_NAME} -type d -empty -delete 2>/dev/null || true
else
    echo "目录 ${REMOTE_PROJECT_NAME} 不存在，创建新目录"
    mkdir -p ${REMOTE_PROJECT_NAME}
fi

echo "解压文件 ${TAR_NAME} 到 ${REMOTE_PROJECT_NAME}（覆盖已存在文件）"
tar -zxf ${TAR_NAME} -C ${REMOTE_PROJECT_NAME}

echo "删除压缩文件 ${TAR_NAME}"
sudo rm -rf ${TAR_NAME}

echo "修改文件权限"
sudo chown -R ec2-user:ec2-user ${REMOTE_PROJECT_NAME}
sudo chmod 664 ${REMOTE_PROJECT_NAME}/*.jar

echo "查看一下当前文件列表"
ls -al ${REMOTE_PROJECT_NAME}

echo "查看是否有指定文件, *.ipdb *.awdb *.dat"
# 使用 ls 检查通配符匹配的文件是否存在
if ls ${REMOTE_PROJECT_NAME}/*.ipdb 1> /dev/null 2>&1; then
    echo "有 ipdb 文件"
else
    echo "未发现 ipdb 文件, 从 ${REMOTE_TOOLS_PATH} 目录下拷贝"
    cp -rf ${REMOTE_TOOLS_PATH}/*.ipdb ${REMOTE_PROJECT_NAME}/ 2>/dev/null || echo "WARNING: ${REMOTE_TOOLS_PATH} 目录下也没有 ipdb 文件"
fi
if ls ${REMOTE_PROJECT_NAME}/*.awdb 1> /dev/null 2>&1; then
    echo "有 awdb 文件"
else
    echo "未发现 awdb 文件, 从 ${REMOTE_TOOLS_PATH} 目录下拷贝"
    cp -rf ${REMOTE_TOOLS_PATH}/*.awdb ${REMOTE_PROJECT_NAME}/ 2>/dev/null || echo "WARNING: ${REMOTE_TOOLS_PATH} 目录下也没有 awdb 文件"
fi
if ls ${REMOTE_PROJECT_NAME}/*.dat 1> /dev/null 2>&1; then
    echo "有 dat 文件"
else
    echo "未发现 dat 文件, 从 ${REMOTE_TOOLS_PATH} 目录下拷贝"
    cp -rf ${REMOTE_TOOLS_PATH}/*.dat ${REMOTE_PROJECT_NAME}/ 2>/dev/null || echo "WARNING: ${REMOTE_TOOLS_PATH} 目录下也没有 dat 文件"
fi

echo "重启容器 ${REMOTE_PROJECT_NAME}"
cd $REMOTE_WORK_PATH
sh docker-compose-restart.sh ${REMOTE_PROJECT_NAME}

echo "检查容器状态 ${REMOTE_PROJECT_NAME}"
if ! checkContainerStatus "${REMOTE_PROJECT_NAME}"; then
    echo "容器启动失败, 发布失败"
    exit 1
fi

echo "发布成功"
